<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;
use App\Project;
use App\Project_Detail;

class ProjectController extends Controller
{
    public function store(Request $request){
        if(Auth::check()){
            $validator =  Validator::make($request->all(),
            [
                'projectname' => 'required|max:30',
                'projectlocation' => 'required|max:20',
                'projectmainimage' => 'required|mimes:jpg,jpeg,jpe,jiff,jif,jfif,png',
                'projectdetailimage.*' => 'mimes:jpg,jpeg,jpe,jiff,jif,jfif,png'
            ],
            [
                'required'=> 'Please enter the :attribute.',
                'projectmainimage.required'=> 'Please upload the project\'s main image.',
                'projectmainimage.mimes'=> 'The project\'s main image must be a file of type: jpg, jpeg, jpe, jiff, jif, jfif, png. Please upload the project\'s main image again.',
                'projectdetailimage.*.mimes'=> 'The project\'s detail images must be a file of type: jpg, jpeg, jpe, jiff, jif, jfif, png. Please upload the project\'s detail images again.'
            ],
            [
                'projectname' => 'project\'s name',
                'projectlocation' => 'project\'s location',
                'projectmainimage' => 'project\'s main image',
                'projectdetailimage.*' => 'project\'s detail image/s'
            ]);

            if($validator->fails()){
                return back()->withErrors($validator)->withInput()->with('failed_add_project', 'yyy');
            }

            $project = new Project();
            $project->project_name = $request->projectname;
            $project->project_location = $request->projectlocation;

            if($request->has('projectmainimage')){
                $project_image_file = time().'_'.$request->projectmainimage->getClientOriginalName();
                $request->projectmainimage->move(\public_path('uploads\projects'), $project_image_file);
                $project->project_image = $project_image_file;
            }
            $project->save();

            if($request->has('projectdetailimage')){
                foreach($request->file('projectdetailimage') as $file){
                    $project_detail_image_file = time().'_'.$file->getClientOriginalName();
                    $file->move(\public_path('uploads\projects_detail'), $project_detail_image_file);
                    $project_detail = new Project_Detail();
                    $project_detail->project_id = $project->id;
                    $project_detail->project_detail_images = $project_detail_image_file;
                    $project_detail->save();
                }                      
            }
            
            return redirect()->back()->with('project_successful_message','The new project is saved successfully!');;
        }
        return redirect('/');
    }

    public function viewProjectDetail($id){
        $projects =  Project::find($id);
        $projects_detail = Project_Detail::select()->where('project_id', $id)->get();
        return response()->json([
            'projects' => $projects, 
            'projects_detail' => $projects_detail
        ]);
    }

    public function viewUpdateData($id){
        $projects =  Project::find($id);
        $projects_detail = Project_Detail::select()->where('project_id', $id)->get();
        return response()->json([
            'projects' => $projects, 
            'projects_detail' => $projects_detail
        ]);
    }

    public function update(Request $request, $id){
        if(Auth::check()){
            $validator =  Validator::make($request->all(),
            [
                'editprojectname' => 'required|max:30',
                'editprojectlocation' => 'required|max:20',
                'editprojectmainimage' => 'mimes:jpg,jpeg,jpe,jiff,jif,jfif,png'
            ],
            [
                'required'=> 'Please enter the :attribute.',
                'editprojectmainimage.mimes'=> 'The project\'s main image must be a file of type: jpg, jpeg, jpe, jiff, jif, jfif, png. Please upload the project\'s main image again.'
            ],
            [
                'editprojectname' => 'project\'s name',
                'editprojectlocation' => 'project\'s location',
                'editprojectmainimage' => 'project\'s main image'
            ]);

            $project = Project::find($id);

            if($validator->fails()){
                $detail_Images=Project_Detail::select()->where('project_id', $id)->get();
                if($request->has('editprojectmainimage')){
                    return redirect()->back()->withErrors($validator)->withInput()->with('failed_update_project', 'update_failed')->with('editprojectdetailimage', $detail_Images);
                }
                return redirect()->back()->withErrors($validator)->withInput()->with('failed_update_project', $project->project_image)->with('editprojectdetailimage', $detail_Images);
            }

            $project->project_name = $request->editprojectname;
            $project->project_location = $request->editprojectlocation;

            if($request->has('editprojectmainimage')){
                $project_image_file = time().'_'.$request->editprojectmainimage->getClientOriginalName();
                $request->editprojectmainimage->move(\public_path('uploads\projects'), $project_image_file);
                $project->project_image = $project_image_file;
            }
            
            $project->save();
            $project_count= $request->editprojectcount;
            return redirect()->back()->with("project_update_successful_message", "The information of the project you want is updated successfully!")->with('editprojectcount', $project_count);
        }
        return redirect('/');

    }

    public function delete($id){
        if(Auth::check()){
            Project::find($id)->delete();

            Project_Detail::where('project_id', $id)->delete();

            return redirect()->back()->with("message", "The information of the project you want is delete successfully!");
        }
        return redirect('/');
    }
}
